<?php

namespace App\Exports;

use Maatwebsite\Excel\Concerns\FromCollection;
use Maatwebsite\Excel\Concerns\WithHeadings;
use Maatwebsite\Excel\Concerns\WithEvents;
use Maatwebsite\Excel\Events\AfterSheet;
use PhpOffice\PhpSpreadsheet\Cell\DataValidation;
use PhpOffice\PhpSpreadsheet\Worksheet\Worksheet;
use Illuminate\Support\Facades\Log;

use App\Models\Vendor;
use App\Models\CarModel;
use App\Models\Category;
use App\Models\City;
use App\Models\Brand;

class SparePartTemplateExport implements FromCollection, WithHeadings, WithEvents
{
    public function collection()
    {
        return collect([]);
    }

    public function headings(): array
    {
        return [
            'name',
            'description',
            'price',
            'vendor',
            'model',
            'category',
            'city',
            'brand',
            'stock',
            'warranty_period',
            'is_new',
            'is_active',
            'images',
        ];
    }

    public function registerEvents(): array
    {
        return [
            AfterSheet::class => function (AfterSheet $event) {
                $sheet = $event->sheet;

                $this->arrayToDropDownListWithIds($sheet, $this->getVendors(), 'D', 1000, 'Vendors');
                $this->arrayToDropDownListWithIds($sheet, $this->getModels(), 'E', 1000, 'Models');
                $this->arrayToDropDownListWithIds($sheet, $this->getCategories(), 'F', 1000, 'Categories');
                $this->arrayToDropDownListWithIds($sheet, $this->getCities(), 'G', 1000, 'Cities');
                $this->arrayToDropDownListWithIds($sheet, $this->getBrands(), 'H', 1000, 'Brands');

                $this->arrayToDropDownList($sheet, ['نعم', 'لا'], 'K', 1000);
                $this->arrayToDropDownList($sheet, ['نعم', 'لا'], 'L', 1000);
            },
        ];
    }

    private function arrayToDropDownListWithIds($sheet, $items, $column, $numberOfRows, $hiddenSheetName)
    {
        if (empty($items)) return;

        $delegateSheet = $sheet->getDelegate();
        $workbook = $delegateSheet->getParent();

        $baseName = $hiddenSheetName;
        $suffix = 1;
        while ($workbook->sheetNameExists($hiddenSheetName)) {
            $hiddenSheetName = $baseName . '_' . $suffix++;
        }

        $hiddenSheet = new Worksheet($workbook, $hiddenSheetName);
        $workbook->addSheet($hiddenSheet);
        $hiddenSheet->setSheetState(Worksheet::SHEETSTATE_HIDDEN);

        foreach ($items as $index => $item) {
            $hiddenSheet->setCellValue('A' . ($index + 1), $item['name']);
            $hiddenSheet->setCellValue('B' . ($index + 1), $item['id']);
        }

        $formula = '=' . $hiddenSheetName . '!$A$1:$A$' . count($items);

        for ($i = 2; $i <= $numberOfRows; $i++) {
            $cell = $delegateSheet->getCell($column . $i);
            $validation = $cell->getDataValidation();
            $validation->setType(DataValidation::TYPE_LIST);
            $validation->setErrorStyle(DataValidation::STYLE_STOP);
            $validation->setAllowBlank(true);
            $validation->setShowInputMessage(true);
            $validation->setShowErrorMessage(true);
            $validation->setShowDropDown(true);
            $validation->setFormula1($formula);
        }
    }

    private function arrayToDropDownList($sheet, $options, $column, $numberOfRows)
    {
        if (empty($options)) return;

        $delegateSheet = $sheet->getDelegate();

        $list = '"' . implode(',', $options) . '"';

        for ($i = 2; $i <= $numberOfRows; $i++) {
            $cell = $delegateSheet->getCell($column . $i);
            $validation = $cell->getDataValidation();
            $validation->setType(DataValidation::TYPE_LIST);
            $validation->setErrorStyle(DataValidation::STYLE_STOP);
            $validation->setAllowBlank(true);
            $validation->setShowInputMessage(true);
            $validation->setShowErrorMessage(true);
            $validation->setShowDropDown(true);
            $validation->setFormula1($list);
        }
    }

    // Data source methods
    private function getVendors(): array
    {
        return Vendor::all()->map(fn($v) => [
            'id' => $v->id,
            'name' => $v->getTranslation('name', 'ar')
        ])->filter(fn($item) => $item['name'])->values()->toArray();
    }

    private function getModels(): array
    {
        return CarModel::all()->map(fn($m) => [
            'id' => $m->id,
            'name' => $m->getTranslation('name', 'ar')
        ])->filter(fn($item) => $item['name'])->values()->toArray();
    }

    private function getCategories(): array
    {
        return Category::all()->map(fn($c) => [
            'id' => $c->id,
            'name' => $c->getTranslation('name', 'ar')
        ])->filter(fn($item) => $item['name'])->values()->toArray();
    }

    private function getCities(): array
    {
        return City::all()->map(fn($city) => [
            'id' => $city->id,
            'name' => is_array($city->name) ? ($city->name['ar'] ?? reset($city->name)) : $city->name
        ])->filter(fn($item) => $item['name'])->values()->toArray();
    }

    private function getBrands(): array
    {
        return Brand::all()->map(fn($brand) => [
            'id' => $brand->id,
            'name' => is_array($brand->name) ? ($brand->name['ar'] ?? reset($brand->name)) : $brand->name
        ])->filter(fn($item) => $item['name'])->values()->toArray();
    }
}
