<?php

namespace App\Http\Controllers\Admin;

use Storage;
use App\Models\Category;
use App\Models\CategoryFilter;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;

class CategoriesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $categories = Category::with('children')->whereNull('parent_id')->get();
        return view('admin.pages.category.index', compact('categories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::whereNull('parent_id')->get();
        return view('admin.pages.category.add', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
     
        try {
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'required|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:6048',
                'parent_id' => 'nullable|exists:categories,id',
                'is_active' => 'required|boolean',
            ]);
            

            $imagePath = null;
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $fileName = time() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('storage/categories'), $fileName);
            
                $imagePath = 'categories/' . $fileName; // relative path
            }

            $category = Category::create([
                'name' => [
                    'ar' => $validated['name_ar'],
                    'en' => $validated['name_en'],
                ],
                'image' => $imagePath,
                'parent_id' => $validated['parent_id'] ?: null,
                'is_active' => $validated['is_active'],
            ]);

            return redirect()->route('admin.categories.index')->with('success', 'تم إنشاء القسم بنجاح');
        } catch (\Exception $e) {
            
            Log::error('Error storing category: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', $e->getMessage().'حدث خطأ أثناء إنشاء القسم. حاول مرة أخرى.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $category = Category::with(['parent', 'children', 'spareParts', 'customPartRequests'])->findOrFail($id);
        return view('admin.pages.category.show', compact('category'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $category = Category::findOrFail($id);
        $categories = Category::whereNull('parent_id')->where('id', '!=', $id)->get();
        return view('admin.pages.category.edit', compact('category', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'required|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:6048',
                'parent_id' => 'nullable|exists:categories,id',
                'is_active' => 'required|boolean',
            ]);

            $category = Category::findOrFail($id);

            $imagePath = $category->image;
           
            if ($request->hasFile('image')) {
                     // remove old one if exists
                    if (!empty($imagePath) && File::exists(public_path($imagePath))) {
                        File::delete(public_path($imagePath));
                    }

                    // save new one
                    $file = $request->file('image');
                    $fileName = time() . '.' . $file->getClientOriginalExtension();
                    $file->move(public_path('storage/categories'), $fileName);

                    $imagePath = 'categories/' . $fileName; 
            }

            $category->update([
                'name' => [
                    'ar' => $validated['name_ar'],
                    'en' => $validated['name_en'],
                ],
                'image' => $imagePath,
                'parent_id' => $validated['parent_id'] ?: null,
                'is_active' => $validated['is_active'],
            ]);

            return redirect()->route('admin.categories.index')->with('success', 'تم تحديث القسم بنجاح');
        } catch (\Exception $e) {
            Log::error('Error updating category: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث القسم. حاول مرة أخرى.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $category = Category::findOrFail($id);

            if ($category->children()->exists() || $category->spareParts()->exists() || $category->customPartRequests()->exists()) {
                return redirect()->back()->with('error', 'لا يمكن حذف القسم لوجود أقسام فرعية أو قطع غيار مرتبطة به');
            }

            if ($category->image && Storage::disk('public')->exists($category->image)) {
                Storage::disk('public')->delete($category->image);
            }

            $category->delete();

            return redirect()->route('admin.categories.index')->with('success', 'تم حذف القسم بنجاح');
        } catch (\Exception $e) {
            Log::error('Error deleting category: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف القسم. حاول مرة أخرى.');
        }
    }
    public function getFiltersByCategory($id)
    {
        $filters = CategoryFilter::where('category_id', $id)->get();

        return response()->json($filters);
    }
}