<?php

namespace App\Http\Controllers\Admin;

use App\Models\Vendor;
use App\Models\Brand;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\File;

class MerchantController extends Controller
{
    /**
     * Display a listing of the merchants.
     */
    public function index()
    {
        $merchants = Vendor::where('type', 0)->with('city')->latest()->get();
        return view('admin.pages.merchant.index', compact('merchants'));
    }

    /**
     * Show the form for creating a new merchant.
     */
    public function create()
    {
        $governates=\App\Models\Governorate::with('cities')->get();
        $cities = \App\Models\City::all();
          $brands=Brand::all();
        return view('admin.pages.merchant.add', compact('cities','governates','brands'));
    }

    /**
     * Store a newly created merchant in storage.
     */
    public function store(Request $request)
    {
        try {
           
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'nullable|string|max:255',
                'email' => 'required|email|unique:vendors,email',
                'password' => 'required|string|min:8|confirmed',
                'phone' => 'required|string|max:20',
                'city_id' => 'required|exists:cities,id',
                'address' => 'required|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'description' => 'nullable|string',
                'is_active' => 'required|boolean',
            ]);

            $vendor = new Vendor();
            $vendor->name = [
                'ar' => $validated['name_ar'],
                'en' => $validated['name_en'] ? $validated['name_en'] : $validated['name_ar'] ,
            ];
            $vendor->email = $validated['email'];
            $vendor->password = Hash::make($validated['password']);
            $vendor->phone = $validated['phone'];
            $vendor->city_id = $validated['city_id'];
            $vendor->address = $validated['address'];
            $vendor->description = $validated['description'];
            $vendor->is_active = $validated['is_active'];
            $vendor->type = 0;

            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $fileName = time() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('storage/vendors/images'), $fileName);
            
                $imagePath = 'vendors/imagess/' . $fileName; 
                $vendor->image =  $imagePath;
            }

            $vendor->save();
            $vendor->brands()->sync($validated['brands'] ?? []);

            return redirect()->route('admin.merchant.index')->with('success', 'تم إضافة التاجر بنجاح');
        } catch (\Exception $e) {
            Log::error('Error storing merchant: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', $e->getMessage().'حدث خطأ أثناء إضافة التاجر. حاول مرة أخرى.');
        }
    }

    /**
     * Show the form for editing the specified merchant.
     */
    public function edit($id)
    {
        $merchant = Vendor::findOrFail($id);
        $cities = \App\Models\City::all();
         $governates=\App\Models\Governorate::with('cities')->get();
        return view('admin.pages.merchant.edit', compact('merchant', 'cities','governates'));
    }

    /**
     * Update the specified merchant in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'nullable|string|max:255',
                'email' => 'required|email|unique:vendors,email,' . $id,
                'password' => 'nullable|string|min:8|confirmed',
                'phone' => 'required|string|max:20',
                'city_id' => 'required|exists:cities,id',
                'address' => 'required|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'description' => 'nullable|string',
                'is_active' => 'required|boolean',
            ]);

            $vendor = Vendor::findOrFail($id);

            $vendor->name = [
                'ar' => $validated['name_ar'],
                'en' => $validated['name_en'] ? $validated['name_en'] : $validated['name_ar'],
            ];
            $vendor->email = $validated['email'];
            if (!empty($validated['password'])) {
                $vendor->password = Hash::make($validated['password']);
            }
            $vendor->phone = $validated['phone'];
            $vendor->city_id = $validated['city_id'];
            $vendor->address = $validated['address'];
            $vendor->description = $validated['description'];
            $vendor->is_active = $validated['is_active'];
            $vendor->type =0;

            if ($request->hasFile('image')) {
                    if (!empty($vendor->image) && File::exists(public_path($vendor->image))) {
                        File::delete(public_path($vendor->image));
                    }

                    $file = $request->file('image');
                    $fileName = time() . '.' . $file->getClientOriginalExtension();
                    $file->move(public_path('storage/vendors/images'), $fileName);

                    $imagePath = 'vendors/images/' . $fileName; 
                    $vendor->image = $imagePath;
            }

            $vendor->save();

            return redirect()->back()->with('success', 'تم تحديث التاجر بنجاح');
        } catch (\Exception $e) {
            Log::error('Error updating merchant: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error',  $e->getMessage().'حدث خطأ أثناء تحديث التاجر. حاول مرة أخرى.');
        }
    }

    /**
     * Remove the specified merchant from storage.
     */
    public function destroy($id,Request $request)
    {
        
        try {
             if($request->input('merchant_ids')){
                  $ids = $request->input('merchant_ids');

                if (!$ids || !is_array($ids) || count($ids) === 0) {
                    return redirect()->back()->with('error', 'لم يتم تحديد أي تاجر للحذف');
                }

                $vendors = \App\Models\Vendor::whereIn('id', $ids)->get();

                if ($vendors->isEmpty()) {
                    return redirect()->back()->with('error', 'لم يتم العثور على التجار المحددين');
                }

                \App\Models\Vendor::whereIn('id', $ids)->delete();

                return redirect()->back()->with('success', 'تم حذف التجار المحددين بنجاح');
             }


            $vendor = Vendor::findOrFail($id);

           

             if (!empty($vendor->image) && File::exists(public_path($vendor->image))) {
                        File::delete(public_path($vendor->image));
                    }

            $vendor->delete();

            return redirect()->back()->with('success', 'تم حذف التاجر بنجاح');
        } catch (\Exception $e) {
            Log::error('Error deleting merchant: ' . $e->getMessage());
            return redirect()->back()->with('error', $e->getMessage() . 'حدث خطأ أثناء حذف التاجر. حاول مرة أخرى.');
        }
    }
    public function bulkDelete(Request $request)
    {
       
       

        
        // Delete merchants
        \App\Models\Vendor::whereIn('id', $ids)->delete();

        return redirect()->back()->with('success', 'تم حذف التجار المحددين بنجاح');
     }
    }