<?php

namespace App\Http\Controllers\Admin;

use App\Models\Vendor;
use App\Models\Inspection;
use App\Models\Specialty;
use App\Models\VendorTimetable;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\File;
use App\Models\Brand;

class ServiceCenterController extends Controller
{
    /**
     * Display a listing of the service centers.
     */
    public function index()
    {
        $serviceCenters = Vendor::where('type',1)->with(['city', 'inspections'])->get();
        return view('admin.pages.service-center.index', compact('serviceCenters'));
    }

    /**
     * Show the form for creating a new service center.
     */
    public function create()
    {
        $cities = \App\Models\City::all();
        $inspections = Inspection::all();
        $specialties = Specialty::all();
        $brands=Brand::all();
        $daysOfWeek = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
        return view('admin.pages.service-center.add', compact('cities', 'inspections', 'specialties', 'daysOfWeek','brands'));
    }

    /**
     * Store a newly created service center in storage.
     */
   public function store(Request $request)
    {
       
        try {
            // ✅ Validation
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'required|string|max:255',
                'email' => 'required|email|unique:vendors,email',
                'password' => 'required|string|min:8|confirmed',
                'phone' => 'required|string|max:11|unique:vendors,phone',
                'city_id' => 'required|exists:cities,id',
                'address' => 'required|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'description' => 'nullable|string',
                'is_active' => 'required|boolean',
                'inspections' => 'nullable|array',
                'inspections.*' => 'exists:inspections,id',
                'brands' => 'nullable|array',
                'brands.*' => 'exists:brands,id',
                'timetables' => 'nullable|array',
                'timetables.*.day_of_week' => 'required|string|in:Sunday,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday',
                'timetables.*.start_time' => 'required|date_format:H:i',
                'timetables.*.end_time' => 'required|date_format:H:i|after:timetables.*.start_time',
                'timetables.*.is_available' => 'required|boolean',
            ]);

            // ✅ Create vendor
            $vendor = new Vendor();
            $vendor->setTranslation('name', 'ar', $validated['name_ar']);
            $vendor->setTranslation('name', 'en', $validated['name_en']);
            $vendor->email = $validated['email'];
            $vendor->password = Hash::make($validated['password']);
            $vendor->phone = $validated['phone'];
            $vendor->city_id = $validated['city_id'];
            $vendor->address = $validated['address'];
            $vendor->description = $validated['description'] ?? null;
            $vendor->is_active = $validated['is_active'];
            $vendor->type = 1;

            // ✅ Handle image upload
            if ($request->hasFile('image')) {
                $file = $request->file('image');
                $fileName = time() . '.' . $file->getClientOriginalExtension();
                $file->move(public_path('storage/vendors/images'), $fileName);

                $imagePath = 'vendors/images/' . $fileName;
                $vendor->image = $imagePath;
            }

            $vendor->save();

            // ✅ Sync relations
            $vendor->inspections()->sync($validated['inspections'] ?? []);
            $vendor->brands()->sync($validated['brands'] ?? []);

            // ✅ Store timetables
            if (!empty($validated['timetables'])) {
                foreach ($validated['timetables'] as $timetable) {
                    $vendor->timetables()->create([
                        'day_of_week' => $timetable['day_of_week'],
                        'start_time' => $timetable['start_time'],
                        'end_time' => $timetable['end_time'],
                        'is_available' => $timetable['is_available'],
                    ]);
                }
            }

            return redirect()
                ->route('admin.service-center.index')
                ->with('success', 'تم إضافة مركز الخدمة بنجاح');

        } catch (ValidationException $e) {
            // ✅ خليه يترمي تاني → Laravel هيتعامل معاه ويعمل redirect مع old() + errors
            throw $e;
        } catch (\Exception $e) {
            // ✅ أي خطأ تاني
            Log::error('Error storing service center: ' . $e->getMessage(), [
                'request' => $request->all()
            ]);
            return redirect()
                ->back()
                ->withInput() // مهم عشان القيم ترجع
                ->with('error', 'حدث خطأ أثناء إضافة مركز الخدمة. حاول مرة أخرى.');
        }
    }

    /**
     * Show the form for editing the specified service center.
     */
    public function edit($id)
    {
        $serviceCenter = Vendor::with(['inspections', 'timetables'])->findOrFail($id);
        $cities = \App\Models\City::all();
         $inspections = Inspection::all();
        $specialties = Specialty::all();
        $daysOfWeek = ['Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday'];
        return view('admin.pages.service-center.edit', compact('serviceCenter', 'cities', 'inspections', 'specialties', 'daysOfWeek'));
    }

    /**
     * Update the specified service center in storage.
     */
    public function update(Request $request, $id)
    {
        try {
             $vendor = Vendor::findOrFail($id);

            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'nullable|string|max:255',
              'email' => [
                    'required',
                    'email',
                    Rule::unique('vendors')->ignore($vendor->id)->whereNull('deleted_at')
                ],
                'password' => 'nullable|string',
             
                'phone' => 'required|string|max:20',
                'city_id' => 'required|exists:cities,id',
                'address' => 'required|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',

                'description' => 'nullable|string',
                'is_active' => 'required|boolean',
                'inspections' => 'nullable|array',
                'inspections.*' => 'exists:inspections,id',
              
                'timetables' => 'nullable|array',
                'timetables.*.day_of_week' => 'nullable|string|in:Sunday,Monday,Tuesday,Wednesday,Thursday,Friday,Saturday',
                'timetables.*.start_time' => 'nullable|date_format:H:i',
                'timetables.*.end_time' => 'nullable|date_format:H:i|after:timetables.*.start_time',
                'timetables.*.is_available' => 'nullable|boolean',
            ]);
            $vendor = Vendor::findOrFail($id);

            $vendor->setTranslation('name', 'ar', $validated['name_ar']);
            $vendor->setTranslation('name', 'en', $validated['name_en']);
            $vendor->email = $validated['email'];
            if (!empty($validated['password'])) {
                $vendor->password = Hash::make($validated['password']);
            }
            $vendor->phone = $validated['phone'];
            $vendor->city_id = $validated['city_id'];
            $vendor->address = $validated['address'];
            $vendor->description = $validated['description'];
            $vendor->is_active = $validated['is_active'];
           // $vendor->type = 2;

           if ($request->hasFile('image')) {
                    if (!empty($vendor->image) && File::exists(public_path($vendor->image))) {
                        File::delete(public_path($vendor->image));
                    }

                    $file = $request->file('image');
                    $fileName = time() . '.' . $file->getClientOriginalExtension();
                    $file->move(public_path('storage/vendors/images'), $fileName);

                    $imagePath = 'vendors/images/' . $fileName; 
                    $vendor->image = $imagePath;
            }

            $vendor->save();

            // Sync inspections and repair issues
            $vendor->inspections()->sync($validated['inspections'] ?? []);
           

            // Update timetables
            $vendor->timetables()->delete(); // Clear existing
            if (!empty($validated['timetables'])) {
                foreach ($validated['timetables'] as $timetable) {
                    if ($timetable['is_available'] == 1) {
                        $vendor->timetables()->create([
                            'day_of_week' => $timetable['day_of_week'],
                            'start_time' => $timetable['start_time'],
                            'end_time' => $timetable['end_time'],
                            'is_available' => $timetable['is_available'],
                        ]);
                    }

                }
            }

            return redirect()->route('admin.service-center.index')->with('success', 'تم تحديث مركز الخدمة بنجاح');
        } catch (\Exception $e) {
            Log::error('Error updating service center: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Remove the specified service center from storage.
     */
    public function destroy($id)
    {
        try {
            $vendor = Vendor::findOrFail($id);

          
           if (!empty($vendor->image) && File::exists(public_path($vendor->image))) {
                        File::delete(public_path($vendor->image));
           }
             $vendor->delete();

            return redirect()->route('admin.service-center.index')->with('success', 'تم حذف مركز الخدمة بنجاح');
        } catch (\Exception $e) {
            Log::error('Error deleting service center: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف مركز الخدمة. حاول مرة أخرى.');
        }
    }
    public function apppintments(Vendor $vendor)
    {
        $appointments = $vendor->appointments()
            ->with(['user', 'inspection'])
            ->latest()
            ->paginate(20);

        return view('admin.pages.service-center.appointments', compact('vendor', 'appointments'));
    }

        public function getInspections($id)
    {
        $inspections = Inspection::where('specialty_id', $id)->get()->map(function ($inspection) {
            return [
                'id' => $inspection->id,
                'title' => $inspection->getTranslation('title', 'ar'),
            ];
        });
        

        return response()->json($inspections);
    }
}