<?php

namespace App\Http\Controllers\Admin;

use App\Models\SiteDetail;
use App\Models\Service;

use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class SiteDetailController extends Controller
{
    /**
     * Display a listing of the site details.
     */
        public function index()
        {
        // Assuming you already have a service or model that returns this structure.
        $response = Http::get('https://app.part-mart.net/api/home')->json(); // Adjust based on your actual API logic

        // Remove "setting" key from response
         unset($response['setting'], $response['message'],$response['section_download_app']);

        // Pass the remaining data
        return view('admin.pages.site-details.index', [
            'site_details' => $response
        ]);
        }

    /**
     * Show the form for creating a new site detail.
     */
    public function create()
    {
        return view('admin.pages.site-details.create');
    }

    /**
     * Store a newly created site detail in storage.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'key' => 'required|string|max:255|unique:site_details,key',
                'title' => 'required|string|max:255',
                'body' => 'required|string',
                'short_description' => 'nullable|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            ]);

            $siteDetail = new SiteDetail();
            $siteDetail->key = $validated['key'];
            $siteDetail->title = $validated['title'];
            $siteDetail->body = $validated['body'];
            $siteDetail->short_description = $validated['short_description'];
            $siteDetail->save();

            if ($request->hasFile('image')) {
                $path = $request->file('image')->store('site-details/images', 'public');
                $siteDetail->image()->create(['path' => $path]);
            }

            return redirect()->route('admin.site-details.index')->with('success', 'تم إضافة التفاصيل بنجاح');
        } catch (\Exception $e) {
            Log::error('Error storing site detail: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة التفاصيل. حاول مرة أخرى.');
        }
    }

    /**
     * Show the form for editing the specified site detail.
     */
    public function edit($id)
    {
        $siteDetail = SiteDetail::with('image')->findOrFail($id);
        return view('admin.pages.site-details.edit', compact('siteDetail'));
    }

    /**
     * Update the specified site detail in storage.
     */
  
public function update(Request $request)
{
    // 1. Update all dynamic sections (not services)
    if ($request->has('sections')) {
        foreach ($request->sections as $id => $sectionData) {
            $section = SiteDetail::find($id);
            if (!$section) continue;

            $section->update([
                'key' => $sectionData['key'],
                'title' => $sectionData['title'] ?? '',
                'short_description' => $sectionData['short_description'] ?? '',
                'body' => $sectionData['body'] ?? '',
            ]);

            // Optional: handle image uploads
            if ($request->hasFile("sections.$id.image")) {
                $imageFile = $request->file("sections.$id.image");
                $path = $imageFile->store('site-details', 'public');
                $section->images()->create(['url' => $path]);
            }

            if ($request->hasFile("sections.$id.image2")) {
                $image2File = $request->file("sections.$id.image2");
                $path2 = $image2File->store('site-details', 'public');
                $section->images()->create(['url' => $path2]);
            }
        }
    }


        if ($request->has('services')) {
           
            foreach ($request->services as $serviceData) {
                if (empty($serviceData['id'])) continue;

                $service = Service::find($serviceData['id']);
               
                if (!$service) continue;

                $updateData = [
                    'short_description' => $serviceData['short_text'] ?? '',
                ];

                // Handle image upload directly to public/services
                if (!empty($serviceData['image']) && $serviceData['image'] instanceof \Illuminate\Http\UploadedFile) {
                    // Remove old image if exists
                    if ($service->image && file_exists(public_path('services/'.$service->image))) {
                        unlink(public_path('services/'.$service->image));
                    }

                    $filename = time().'_'.$serviceData['image']->getClientOriginalName();
                    $serviceData['image']->move(public_path('storage/services'), $filename);

                    $updateData['image'] =$filename; // save relative path
                }

                $service->update($updateData);
            }
        }

        // Handle new service
        if ($request->has('new_service') && !empty($request->new_service['short_text'])) {
            $newData = [
                'short_description' => $request->new_service['short_text'],
            ];

            if (!empty($request->new_service['image']) && $request->new_service['image'] instanceof \Illuminate\Http\UploadedFile) {
                $filename = time().'_'.$request->new_service['image']->getClientOriginalName();
                $request->new_service['image']->move(public_path('services'), $filename);

                $newData['image'] = 'services/'.$filename;
            }

            Service::create($newData);
        }

    return redirect()->back()->with('success', 'تم التحديث بنجاح');
}

    /**
     * Remove the specified site detail from storage.
     */
    public function destroy($id)
    {
        try {
            $siteDetail = SiteDetail::findOrFail($id);

            if ($siteDetail->image && Storage::disk('public')->exists($siteDetail->image->path)) {
                Storage::disk('public')->delete($siteDetail->image->path);
                $siteDetail->image()->delete();
            }

            $siteDetail->delete();

            return redirect()->route('admin.site-details.index')->with('success', 'تم حذف التفاصيل بنجاح');
        } catch (\Exception $e) {
            Log::error('Error deleting site detail: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف التفاصيل. حاول مرة أخرى.');
        }
    }
}