<?php

namespace App\Http\Controllers\Admin;

use App\Models\ModelSlider;
use Illuminate\Http\Request;
use App\Models\Vendor;
use App\Models\SparePart;
use App\Http\Controllers\Controller;
use Carbon\Carbon;

class SliderController extends Controller
{
    public function index()
    {
        $now = Carbon::now();
        $activeOffers = SparePart::whereNotNull('offer_starts_at')
                ->where(function ($query) use ($now) {
                    $query->whereNull('offer_ends_at')
                        ->orWhere('offer_ends_at', '>=', $now);
                })
                ->with('images')
                ->latest()
                ->get();
        $sliders = ModelSlider::all();
        $vendors = Vendor::with('brands.models')->where('type',1)->get(); // or just Vendor::all() if simple
        $models = \App\Models\Model::all();
        return view('admin.pages.banner.index', compact('sliders','models','vendors','activeOffers'));
    }
    public function create()
    {
        return view('admin.pages.banner.create');
    }

    // Store a newly created slider in the database
   public function store(Request $request)
    {
        $request->validate([
            'image' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'title_ar' => 'required|string|max:255',
            'title_en' => 'required|string|max:255',
            'model_id' => 'nullable|exists:models,id',
            'vendor_id' => 'nullable|exists:vendors,id',
            'order' => 'nullable|integer',
            'is_active' => 'required|boolean',
            'offer_id'=>'nullable',

        ]);
       

        $imagePath = $request->file('image')->store('sliders', 'public');
         if ($request->hasFile('image')) {
                     $fileName = time() . '.' . $request->image->getClientOriginalExtension();
                     $request->image->move(public_path('storage/sliders'), $fileName);
                    $imagePath = 'sliders/' . $fileName; 
        }

        ModelSlider::create([
            'title' => [
                'ar' => $request->title_ar,
                'en' => $request->title_en,
            ],
            'image_url' => 'storage/' . $imagePath,
            'model_id' => $request->model_id,
            'vendor_id' => $request->vendor_id ? $request->vendor_id : null,
            'order' => $request->order ?? 0,
            'spare_part_id'=>$request->offer_id ? $request->offer_id : null,
            'is_active' => $request->is_active,
        ]);

        return redirect()->route('admin.sliders.index')->with('success', 'تم إنشاء السلايدر بنجاح');
    }

    // Show the form for editing a slider
    public function edit( $id)
    {
        $slider=ModelSlider::find($id);
      
        return view('admin.pages.banner.edit', compact('slider'));
    }

    // Update the specified slider in the database
    public function update(Request $request, $id)
    {
          $slider=ModelSlider::find($id);
       
        $request->validate([
                'title.ar' => 'required|string|max:255',
                'title.en' => 'required|string|max:255',
                'order' => 'nullable|integer',
                'is_active' => 'required|boolean',
                'image_url' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            ]);

           
          if ($request->hasFile('image_url')) {
                if ($slider->image_url && file_exists(public_path('storage/' . $slider->image_url))) {
                    unlink(public_path('storage/' . $slider->image_url));
                }

                $fileName = time() . '.' . $request->image_url->getClientOriginalExtension();
                $request->image_url->move(public_path('storage/sliders'), $fileName);
                $slider->image_url = 'storage/sliders/' . $fileName;
            }

            
            $slider->title = $request->input('title');
            $slider->order = $request->input('order', 0);
            $slider->is_active = $request->boolean('is_active');

            $slider->save();

        return redirect()->route('admin.sliders.index')->with('success', 'Slider updated successfully');
    }

    // Remove the specified slider from the database
    public function destroy($slider)
    {
        ModelSlider::find($slider)->delete();
        return redirect()->route('admin.sliders.index')->with('success', 'Slider deleted successfully');
    }

    public function getVendorModels(Request $request)
    {
       $vendor = Vendor::with('brands.models')->findOrFail($request->vendor_id);

        $models = $vendor->brands->flatMap(function ($brand) {
            return $brand->models; 
        })->unique('id')->values(); 

       
       /*  return response()->json([
            'models' => $models->map(function ($model) {
                return [
                    'id' => $model->id,
                    'name' => $model->name,
                ];
            }),
        ]); */
        return response()->json(
            $models->map(fn($model) => [
                'id' => $model->id,
                'name' => $model->getTranslation('name','ar'),
            ])
        );
    }

}