<?php

namespace App\Http\Controllers\Admin;

use App\Models\Store;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class StoreController extends Controller
{
    public function index()
    {
        $stores = Store::all();
        return view('admin.pages.store.index', compact('stores'));

    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name_ar' => 'required|string|max:255',
            'name_en' => 'required|string|max:255',
            'email' => 'required|email|unique:stores,email',
            'password' => 'required|string|min:8',
            'image' => 'nullable|image|mimes:jpeg,png,jpg',
            'description_ar' => 'nullable|string',
            'description_en' => 'nullable|string',
        ]);
        $store = new Store();
        $store->setTranslations('name', [
            'ar' => $data['name_ar'],
            'en' => $data['name_en'],
        ]);
        $store->setTranslations('description', [
            'ar' => $data['description_ar'] ?? '',
            'en' => $data['description_en'] ?? '',
        ]);
        $store->email = $data['email'];
        $store->password = Hash::make($data['password']);

        if ($request->hasFile('image')) {
            $store->image = $request->file('image')->store('stores', 'public');
        }
        if ($request->hasFile('logo')) {
            $store->logo = $request->file('logo')->store('stores', 'public');
        }
        $store->save();
        return redirect()->route('admin.stores.index')->with('success', 'تم إضافة المتجر بنجاح');
    }
    public function update(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'required|string|max:255',
                'description_ar' => 'required|string',
                'description_en' => 'required|string',
                'image' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
                'logo' => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
            ]);

            $store = Store::findOrFail($id);

            $store->setTranslations('name', [
                'ar' => $validated['name_ar'],
                'en' => $validated['name_en'],
            ]);

            $store->setTranslations('description', [
                'ar' => $validated['description_ar'],
                'en' => $validated['description_en'],
            ]);

            if ($request->hasFile('image')) {
                if ($store->image && Storage::exists('public/' . $store->image)) {
                    Storage::delete('public/' . $store->image);
                }
                $store->image = $request->file('image')->store('stores/images', 'public');
            }

            if (isset($request->password) && !empty($request->password)) {
                $store->password = Hash::make($request->password);
            }

            // Handle logo upload
            if ($request->hasFile('logo')) {
                if ($store->logo && Storage::exists('public/' . $store->logo)) {
                    Storage::delete('public/' . $store->logo);
                }
                $store->logo = $request->file('logo')->store('stores/logos', 'public');
            }

            // Save the store
            $store->save();

            // Redirect with success message
            return redirect()->route('admin.stores.index')->with('success', 'Store updated successfully!');
        } catch (\Exception $e) {
            return back()->with('error', 'An error occurred while updating the store. Please try again later.');
        }
    }


    public function destroy($id)
    {
        $store = Store::findOrFail($id);

        if ($store->image && Storage::exists('public/' . $store->image)) {
            Storage::delete('public/' . $store->image);
        }

        if ($store->logo && Storage::exists('public/' . $store->logo)) {
            Storage::delete('public/' . $store->logo);
        }

        $store->delete();

        return redirect()->route('admin.stores.index')->with('success', 'Store deleted successfully!');
    }

}