<?php

namespace App\Http\Controllers\Admin;

use Storage;
use App\Models\Category;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Log;

class SubCategoryController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $categories = Category::with('parent')->whereNotNull('parent_id')->get();
        $pcategories = Category::whereNull('parent_id')->get();
        return view('admin.pages.sub_category.index', compact('categories', 'pcategories'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $categories = Category::whereNull('parent_id')->get();
        return view('admin.pages.sub_category.add', compact('categories'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'required|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'parent_id' => 'required|exists:categories,id',
                'is_active' => 'required|boolean',
            ]);

            $imagePath = null;
            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('categories', 'public');
            }

            Category::create([
                'name' => [
                    'ar' => $validated['name_ar'],
                    'en' => $validated['name_en'],
                ],
                'image' => $imagePath,
                'parent_id' => $validated['parent_id'],
                'is_active' => $validated['is_active'],
            ]);

            return redirect()->route('admin.subCategories.index')->with('success', 'تم إنشاء القسم الفرعي بنجاح');
        } catch (\Exception $e) {
            Log::error('Error storing subcategory: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', 'حدث خطأ أثناء إنشاء القسم الفرعي. حاول مرة أخرى.');
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $category = Category::with(['parent', 'children', 'spareParts', 'customPartRequests'])->findOrFail($id);
        return view('admin.pages.sub_category.show', compact('category'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $category = Category::findOrFail($id);
        $categories = Category::whereNull('parent_id')->where('id', '!=', $id)->get();
        return view('admin.pages.sub_category.edit', compact('category', 'categories'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'name_ar' => 'required|string|max:255',
                'name_en' => 'required|string|max:255',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
                'parent_id' => 'required|exists:categories,id',
                'is_active' => 'required|boolean',
            ]);

            $category = Category::findOrFail($id);

            $imagePath = $category->image;
            if ($request->hasFile('image')) {
                if ($imagePath && Storage::disk('public')->exists($imagePath)) {
                    Storage::disk('public')->delete($imagePath);
                }
                $imagePath = $request->file('image')->store('categories', 'public');
            }

            $category->update([
                'name' => [
                    'ar' => $validated['name_ar'],
                    'en' => $validated['name_en'],
                ],
                'image' => $imagePath,
                'parent_id' => $validated['parent_id'],
                'is_active' => $validated['is_active'],
            ]);

            return redirect()->route('admin.subCategories.index')->with('success', 'تم تحديث القسم الفرعي بنجاح');
        } catch (\Exception $e) {
            Log::error('Error updating subcategory: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث القسم الفرعي. حاول مرة أخرى.');
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        try {
            $category = Category::findOrFail($id);

            if ($category->children()->exists() || $category->spareParts()->exists() || $category->customPartRequests()->exists()) {
                return redirect()->back()->with('error', 'لا يمكن حذف القسم الفرعي لوجود أقسام فرعية أو قطع غيار مرتبطة به');
            }

            if ($category->image && Storage::disk('public')->exists($category->image)) {
                Storage::disk('public')->delete($category->image);
            }

            $category->delete();

            return redirect()->route('admin.subCategories.index')->with('success', 'تم حذف القسم الفرعي بنجاح');
        } catch (\Exception $e) {
            Log::error('Error deleting subcategory: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف القسم الفرعي. حاول مرة أخرى.');
        }
    }
}