<?php

namespace App\Http\Controllers\Admin;

use App\Models\User;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class UserController extends Controller
{
    /**
     * Display a listing of the users.
     */
    public function index()
    {
        $users = User::with('city')->get();
        return view('admin.pages.users.index', compact('users'));
    }

    /**
     * Show the form for creating a new user.
     */
    public function create()
    {
        $cities = \App\Models\City::all();
        return view('admin.pages.users.create', compact('cities'));
    }

    /**
     * Store a newly created user in storage.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'full_name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email',
                'password' => 'required|string|min:8|confirmed',
                'phone' => 'required|string|max:20',
                'city_id' => 'required|exists:cities,id',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'email_verified' => 'required|boolean',
            ]);

            $user = new User();
            $user->full_name = $validated['full_name'];
            $user->email = $validated['email'];
            $user->password = Hash::make($validated['password']);
            $user->phone = $validated['phone'];
            $user->city_id = $validated['city_id'];
            $user->email_verified = $validated['email_verified'];

            if ($request->hasFile('image')) {
                $user->image = $request->file('image')->store('users/images', 'public');
            }

            $user->save();

            return redirect()->route('admin.users.index')->with('success', 'تم إضافة المستخدم بنجاح');
        } catch (\Exception $e) {
            Log::error('Error storing user: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', 'حدث خطأ أثناء إضافة المستخدم. حاول مرة أخرى.');
        }
    }

    /**
     * Show the form for editing the specified user.
     */
    public function edit($id)
    {
        $user = User::findOrFail($id);
        $cities = \App\Models\City::all();
        return view('admin.pages.users.edit', compact('user', 'cities'));
    }

    /**
     * Update the specified user in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'full_name' => 'required|string|max:255',
                'email' => 'required|email|unique:users,email,' . $id,
                'password' => 'nullable|string|min:8|confirmed',
                'phone' => 'required|string|max:20',
                'city_id' => 'required|exists:cities,id',
                'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'email_verified' => 'required|boolean',
            ]);

            $user = User::findOrFail($id);
            $user->full_name = $validated['full_name'];
            $user->email = $validated['email'];
            if (!empty($validated['password'])) {
                $user->password = Hash::make($validated['password']);
            }
            $user->phone = $validated['phone'];
            $user->city_id = $validated['city_id'];
            $user->email_verified = $validated['email_verified'];

            if ($request->hasFile('image')) {
                if ($user->image && Storage::disk('public')->exists($user->image)) {
                    Storage::disk('public')->delete($user->image);
                }
                $user->image = $request->file('image')->store('users/images', 'public');
            }

            $user->save();

            return redirect()->route('admin.users.index')->with('success', 'تم تحديث المستخدم بنجاح');
        } catch (\Exception $e) {
            Log::error('Error updating user: ' . $e->getMessage(), ['request' => $request->all()]);
            return redirect()->back()->with('error', 'حدث خطأ أثناء تحديث المستخدم. حاول مرة أخرى.');
        }
    }

    /**
     * Remove the specified user from storage.
     */
    public function destroy($id)
    {
        try {
            $user = User::findOrFail($id);

            if (
                $user->carsForSale()->exists() ||
                $user->customPartRequests()->exists() ||
                $user->carts()->exists() ||
                $user->orders()->exists() ||
                $user->towRequests()->exists() ||
                $user->userCars()->exists() ||
                $user->notifications()->exists()
            ) {
                return redirect()->back()->with('error', 'لا يمكن حذف المستخدم لوجود بيانات مرتبطة به');
            }

            if ($user->image && Storage::disk('public')->exists($user->image)) {
                Storage::disk('public')->delete($user->image);
            }

            $user->delete();

            return redirect()->route('admin.users.index')->with('success', 'تم حذف المستخدم بنجاح');
        } catch (\Exception $e) {
            Log::error('Error deleting user: ' . $e->getMessage());
            return redirect()->back()->with('error', 'حدث خطأ أثناء حذف المستخدم. حاول مرة أخرى.');
        }
    }
}