<?php
namespace App\Http\Controllers\MobileApp;

use Exception;
use Carbon\Carbon;
use App\Models\City;
use App\Models\User;
use App\Models\ContactUs;
use App\Mail\PasswordReset;
use App\Models\Governorate;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Http\Resources\UserResource;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Http\Resources\ContactUsResource;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\GovernorateResource;
use App\Http\Resources\ContactUsIconResource;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Http;

class AuthController extends Controller
{
    /**
     * Constructor to set the locale based on Accept-Language header
     */
    public function __construct(Request $request)
    {
        $locale = $request->header('Accept-Language', 'ar');
        App::setLocale($locale);
    }
    public function changePassword(Request $request)
    {
         
    
        try {
            
            $request->validate([
                'current_password' => 'required',
                'new_password' => 'required|string|min:8|confirmed',
            ]);

           
            $user = Auth::user();

          
            if (!Hash::check($request->current_password, $user->password)) {
                throw ValidationException::withMessages([
                    'current_password' => ['The provided current password is incorrect.'],
                ]);
            }

            // Update the password
            $user->password = Hash::make($request->new_password);
            $user->save();

            return response()->json(['message' => 'Password changed successfully.'], 200);
        } catch (ValidationException $e) {
            return response()->json([
                'message' => __('messages.incorrect_password'),
                'errors' => $e->errors(),
            ], 422);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'An error occurred while changing the password.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    /**
     * Register a new user
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'full_name' => 'required|string|max:255',
            'phone' => 'required|string|max:20|unique:users',
            'email' => 'nullable|string|email|max:255|unique:users',
            'password' => 'required|string|min:8|confirmed',
            'city_id' => 'required|exists:cities,id',
            'device_token' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = User::create([
            'full_name' => $request->full_name,
            'phone' => $request->phone,
            'email' => $request->email,
            'password' => Hash::make($request->password),
            'city_id' => $request->city_id,
            'email_verification_code' => Str::random(40),
            'email_verified' => false,
            'fcm_token' => $request->device_token, // Save device token
        ]);

        $token = $user->createToken('mobile-app')->plainTextToken;

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.user_registered'),
            'user' => $user,
            'token' => $token,
        ], 201);
    }
    /**
     * Login user and issue token
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'phone' => 'required',
            'password' => 'required|string',
            'device_token' => 'required|string', // Add device token validation
        ]);
        $fieldType = filter_var($request->input('phone'), FILTER_VALIDATE_EMAIL) ? 'email' : 'phone';
         

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        if (!Auth::attempt([$fieldType => $request->input('phone'), 'password' =>  $request->input('password')])) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.invalid_credentials'),
            ], 401);
        }

        $user = Auth::user();
        $user->update(['fcm_token' => $request->device_token]); // Save device token
        $token = $user->createToken('mobile-app')->plainTextToken;

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.user_logged_in'),
            'user' => $user,
            'token' => $token,
            'has_car'=>$user->userCars->count() > 0 ? true : false ,

           
        ], 200);
    }

    /**
     * Logout user and revoke token
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout(Request $request)
    {
        $user = $request->user();

        if ($user && $user->currentAccessToken()) {
            $user->currentAccessToken()->delete();
        }

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.user_logged_out'),
        ], 200);
    }


    /**
     * Send password reset link/code
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function forgotPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|exists:users,email',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = User::where('email', $request->email)->first();
        $token = mt_rand(100000, 999999);

        DB::table('password_reset_tokens')->updateOrInsert(
            ['email' => $request->email],
            [
                'token' => $token,
                'created_at' => Carbon::now(),
            ]
        );

        // Send OTP via email
        Mail::to($user->email)->send(new PasswordReset($token, $user));

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.password_reset_otp_sent'),
            'token' => $token,
            'email' => $request->email
        ], 200);
    }

    /**
     * Reset password using token
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function resetPassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|exists:users,email',
            'token' => 'required|string',
            'password' => 'required|string|min:8|confirmed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $passwordReset = DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->where('token', $request->token)
            ->first();

        if (!$passwordReset || Carbon::parse($passwordReset->created_at)->addHour()->isPast()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.invalid_token'),
            ], 400);
        }

        $user = User::where('email', $request->email)->first();
        $user->update([
            'password' => Hash::make($request->password),
        ]);

        DB::table('password_reset_tokens')->where('email', $request->email)->delete();

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.password_reset_success'),
        ], 200);
    }

    /**
     * Save device token for push notifications
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function saveDeviceToken(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'token' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = Auth::user();
        $user->update(['device_token' => $request->token]);

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.device_token_saved'),
        ], 200);
    }

    /**
     * Update user profile
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateProfile(Request $request)
    {
        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            $validator = Validator::make($request->all(), [
                'full_name' => 'sometimes|string|max:255',
                'phone' => ['sometimes', 'string', 'max:20', 'unique:users,phone,' . $user->id],
                'email' => ['sometimes', 'string', 'email', 'max:255', 'unique:users,email,' . $user->id],
                'city_id' => 'sometimes|exists:cities,id',
                'image' => 'sometimes|image|mimes:jpg,png,jpeg|max:2048',
                'device_token' => 'sometimes|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.validation_failed'),
                    'errors' => $validator->errors(),
                ], 422);
            }

            $data = $validator->validated();

            // Handle image upload
            if ($request->hasFile('image')) {
                $data['image'] = $request->file('image')->store('profiles', 'public');
            }

            DB::beginTransaction();

            $user->update($data);

            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.profile_updated'),
                'data' => new UserResource($user)
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating user profile', [
                'user_id' => $user->id ?? null,
                'data' => $request->all(),
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_update_profile'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }
    public function userData()
    {
        return response()->json([
            'status' => 'success',
            'message' => trans('messages.profile_updated'),
            'data' => new UserResource(auth('sanctum')->user()),
            'governorates' => GovernorateResource::collection(Governorate::with('cities')->get())
        ], 200);
    }
    public function deleteAccount(Request $request)
    {
        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            // Log deletion attempt
            Log::info('Account deletion attempt', [
                'user_id' => $user->id,
                'email' => $user->email,
                'locale' => app()->getLocale(),
            ]);

            DB::transaction(function () use ($user) {
                // Soft delete related records
                $user->userCars()->delete();
                //  $user->carsForSale()->delete();
                //  $user->customPartRequests()->delete();
                $user->carts()->delete();
                $user->orders()->delete();
                $user->towRequests()->delete();
                $user->notifications()->delete();

                // Revoke all tokens
                $user->tokens()->delete();

                // Soft delete the user
                $user->delete();
            });

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.account_deleted'),
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error deleting user account', [
                'user_id' => auth('sanctum')->user()->id ?? null,
                'email' => auth('sanctum')->user()->email ?? null,
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_delete_account'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }
    public function getSocial()
    {
        $contact = ContactUs::select([
            'email',
            'phone',
            'facebook',
            'whatsapp',
            'instagram',
            'tiktok',
            'youtube'
        ])->first();

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.social_data_retrieved'),
            'data' => new ContactUsResource($contact)
        ], 200);
    }

    public function getIconWhatsapp()
    {
        $contact = ContactUs::select(['ac_icon_whatsapp'])->first();

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.whatsapp_icon_retrieved'),
            'data' => new ContactUsIconResource($contact)
        ], 200);
    }

    public function verifyOtp(Request $request)
    {
        // Validate input
        $validator = Validator::make($request->all(), [
            'email' => 'required|string|email|exists:users,email',
            'otp' => 'required|string|size:6',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $resetToken = DB::table('password_reset_tokens')
            ->where('email', $request->email)
            ->where('token', $request->otp)
            ->first();

        if (!$resetToken) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.invalid_otp'),
            ], 400);
        }

        // Check OTP expiration (60 minutes)
        if (Carbon::parse($resetToken->created_at)->addMinutes(60)->isPast()) {

            return response()->json([
                'status' => 'error',
                'message' => trans('messages.expired_otp'),
            ], 400);
        }

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.otp_verified'),
        ], 200);
    }
   
   
    public function deleteByPhone($phone)
    {
        $user = User::where('phone', $phone)->first();

        if (!$user) {
            return response()->json([
                'status' => false,
                'message' => 'User not found',
            ], 200);
        }

        $user->delete();

        return response()->json([
            'status' => true,
            'message' => 'User deleted successfully',
        ], 200);
    }
   public function googelLogin(Request $request)
{
    try {
        $request->validate([
            'access_token' => 'required|string',
        ]);

        $accessToken = $request->access_token;

        $googleUser = Http::withToken($accessToken)
            ->get('https://www.googleapis.com/oauth2/v2/userinfo')
            ->throw()
            ->json();

        if (!isset($googleUser['email'])) {
            return response()->json(['error' => 'Access token غير صالح'], 401);
        }

        $peopleData = Http::withToken($accessToken)
            ->get('https://people.googleapis.com/v1/people/me', [
                'personFields' => 'phoneNumbers,addresses',
            ])
            ->throw()
            ->json();

        $phone = $peopleData['phoneNumbers'][0]['value'] ?? null;
        $city  = $peopleData['addresses'][0]['city'] ?? null;

        $user = User::updateOrCreate(
            ['email' => $googleUser['email']],
            [
                'full_name' => $googleUser['name'] ?? '',
                'avatar'    => $googleUser['picture'] ?? null,
                'phone'     => $phone,
                'city'      => $city,
                
               'fcm_token' =>$request->fcm_token,
            ]
        );
       
        if ($user->wasRecentlyCreated) {
            event(new \App\Events\GoogelLogin($user));
        }

        $token = $user->createToken('auth_token')->plainTextToken;

        return response()->json([
            'token' => $token,
            'user'  => $user,
            'has_car'=>$user->userCars->count() > 0 ? true : false ,
            'has_password'=>$user->password == null ? false : true,
        ]);

    } catch (\Illuminate\Http\Client\RequestException $e) {
        $status = $e->response ? $e->response->status() : 500;

        if ($status === 401) {
            return response()->json([
                'error' => 'Access token غير صالح'
            ], 401);
        }

        return response()->json([
            'error' => 'Google API request failed',
            'message' => $e->getMessage(),
            'response' => $e->response ? $e->response->json() : null
        ], $status);

    } catch (\Exception $e) {
        return response()->json([
            'error' => 'حدث خطأ غير متوقع',
            'message' => $e->getMessage()
        ], 500);
    }
}

}