<?php
namespace App\Http\Controllers\MobileApp;

use App\Models\Cart;
use App\Models\CartItem;
use App\Models\SparePart;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Resources\CartResource;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Services\FirebaseNotificationService;

class CartController extends Controller
{
    protected $notificationService;

    /**
     * Constructor to set locale and initialize notification service
     */
    public function __construct(FirebaseNotificationService $notificationService, Request $request)
    {
        $this->notificationService = $notificationService;
        $locale = $request->header('Accept-Language', 'en');
        \Illuminate\Support\Facades\App::setLocale($locale);
    }
    // public function __construct(Request $request)
    // {
    //     $locale = $request->header('Accept-Language', 'ar');
    //     \Illuminate\Support\Facades\App::setLocale($locale);
    // }
    /**
     * Get the user's cart
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        $user = Auth::user();
        $cart = Cart::firstOrCreate(['user_id' => $user->id]);

        $cart->load('items.sparePart');

        $total = $cart->items->sum(function ($item) {
            return $item->quantity * $item->price;
        });

        return response()->json([
            'status' => 'success',
            'message' => trans('messages.cart_retrieved'),
            'data' => [
                'cart' => new CartResource($cart),
                'total' => $total,
            ],
        ], 200);
    }

    /**
     * Add item to cart
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function add(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'spare_part_id' => 'required|exists:spare_parts,id',
            'quantity' => 'required|integer|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = Auth::user();
        $sparePart = SparePart::findOrFail($request->spare_part_id);

        if ($sparePart->stock < $request->quantity) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.insufficient_stock'),
            ], 400);
        }

        $cart = Cart::firstOrCreate(['user_id' => $user->id]);

        $cartItem = $cart->items()->where('spare_part_id', $request->spare_part_id)->first();

        DB::beginTransaction();
        try {
            if ($cartItem) {
                if($cartItem->quantity + $request->quantity   > $sparePart->stock){
                    throw new \Exception( trans('messages.insufficient_stock'));
                }
                $cartItem->update([
                    'quantity' => $cartItem->quantity + $request->quantity,
                    'price' => $sparePart->price,
                ]);
            } else {
                $cart->items()->create([
                    'spare_part_id' => $request->spare_part_id,
                    'quantity' => $request->quantity,
                    'price' => $sparePart->price,
                ]);
            }

            // Send notification
            $this->notificationService->sendNotification(
                $user,
                trans('messages.item_added_to_cart'),
                trans('messages.item_added_to_cart_notification', ['name' => $sparePart->name, 'quantity' => $request->quantity]),
                ['cart_id' => $cart->id, 'spare_part_id' => $sparePart->id]
            );

            DB::commit();
            $cart->load('items.sparePart');

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.item_added_to_cart'),
                'data' => new CartResource($cart),
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_add_item'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update cart item quantity
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'quantity' => 'required|integer|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.validation_failed'),
                'errors' => $validator->errors(),
            ], 422);
        }

        $user = Auth::user();
        $cart = Cart::where('user_id', $user->id)->firstOrFail();
        $cartItem = $cart->items()->findOrFail($id);
        $sparePart = $cartItem->sparePart;

        if ($sparePart->stock < $request->quantity) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.insufficient_stock'),
            ], 400);
        }

        DB::beginTransaction();
        try {
            $cartItem->update([
                'quantity' => $request->quantity,
                'price' => $sparePart->price,
            ]);

            // Send notification
            $this->notificationService->sendNotification(
                $user,
                trans('messages.cart_item_updated'),
                trans('messages.cart_item_updated_notification', ['name' => $sparePart->name, 'quantity' => $request->quantity]),
                ['cart_id' => $cart->id, 'spare_part_id' => $sparePart->id]
            );

            DB::commit();
            // $cart->load('items.sparePart');

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.cart_item_updated'),
                'data' => new CartResource($cart),
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_update_item'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Remove item from cart
     *
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function remove($id)
    {
        $user = Auth::user();
        $cart = Cart::where('user_id', $user->id)->firstOrFail();
        $cartItem = $cart->items()->findOrFail($id);
        $sparePart = $cartItem->sparePart;

        DB::beginTransaction();
        try {
            $cartItem->delete();

            // Send notification
            $this->notificationService->sendNotification(
                $user,
                trans('messages.item_removed_from_cart'),
                trans('messages.item_removed_from_cart_notification', ['name' => $sparePart->name]),
                ['cart_id' => $cart->id, 'spare_part_id' => $sparePart->id]
            );

            DB::commit();
            $cart->load('items.sparePart');

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.item_removed_from_cart'),
                'data' => new CartResource($cart), // Fixed to use CartResource
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_remove_item'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}