<?php
namespace App\Http\Controllers\MobileApp;

use App\Models\Brand;
use App\Models\Vendor;
use App\Models\Category;
use App\Models\Notification;
use Illuminate\Http\Request;
use App\Models\CustomPartRequest;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Http\Resources\BrandModelResource;
use Illuminate\Support\Facades\Cache;
use App\Http\Resources\VendorResource;
use App\Http\Resources\CategoryResource;
use Illuminate\Validation\ValidationException;
use App\Http\Resources\CustomPartRequestResource;
use App\Http\Requests\MobileApi\StoreCustomPartRequest;
use App\Notifications\NewCustomPartRequestNotification;
use App\Services\FirebaseNotificationService;
use App\Services\VendorNotificationService;
use App\Traits\ApiTrait;
use App\Http\Resources\SparePartResource;
use App\Models\SparePartOffer;
use App\Http\Resources\CustomPartRequestItemResource;
use App\Models\Coupon;

class CustomPartRequestController extends Controller
{
    use ApiTrait;
    /**
     * Constructor to set locale
     */
    public function __construct(Request $request)
    {
        $locale = $request->header('Accept-Language', 'ar');
        \Illuminate\Support\Facades\App::setLocale($locale);
    }

    public function store(Request $request)
    {
        try {
         
            $validated = $request->validate([
                'model_id' => 'required|exists:models,id',
                'category_id' => 'required|exists:categories,id',
                'city_id'=>'required|exists:cities,id',
                'description' => 'nullable|string',
                'address' => 'nullable|string',
                'items' => 'required|array|min:1',
                'items.*.name' => 'required|string',
                'items.*.notes' => 'nullable|string',
                'items.*.quantity' => 'required|integer|min:1',
                'items.*.images' => 'nullable|array',
                'items.*.images.*' => 'nullable|file|mimes:jpeg,png,jpg,gif,mp4,mov,webm,mkv,avi|max:6048',
            ]);
           

           
            $userId = Auth::guard('sanctum')->check() ? Auth::guard('sanctum')->user()->id : null;
            $tokenId = null;
            if (!$userId) {
                $tokenId = $request->header('tokenID');
                if (!$tokenId) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'يجب توفير tokenID في الهيدر للمستخدمين الغير مصرح لهم',
                    ], 400);
                }
            }

          
            $requestData = [
                'model_id' => $validated['model_id'],
                'category_id' => $validated['category_id'],
                'description' => $validated['description'],
                'city_id'=>$validated['city_id'],
                'status' => 'pending',
                'address' => $validated['address'],
                'user_id' => $userId,
                'code' => $tokenId,
            ];

            DB::beginTransaction();

           
            $customPartRequest = CustomPartRequest::create($requestData);

        
           foreach ($validated['items'] as $index => $itemData) {
            
                $item = $customPartRequest->items()->create([
                    'name' => $itemData['name'],
                    'quantity' => $itemData['quantity'],
                    'notes' => $itemData['notes'] ?? null,
                ]);

                if ($request->hasFile("items.{$index}.images")) {
                  
                    foreach ($request->file("items.{$index}.images") as $imageFile) {
                       
                        $imagePath = $imageFile->store('images/parts', 'public');

                        $item->images()->create([
                            'path' => $imagePath,
                        ]);
                    }
                }
            }

            // Send notification to vendor
           // $vendor = $customPartRequest->vendor;
            // if ($vendor) {
            //     $vendor->notify(new NewCustomPartRequestNotification([
            //         'custom_part_request_id' => $customPartRequest->id,
            //         'message' => trans('messages.custom_part_created'),
            //     ]));
            // }
            $vendorNotifier = new VendorNotificationService();

            $vendorNotifier->notifyVendorsByCityAndBrand(
                $validated['city_id'],
                $validated['model_id'],
                $customPartRequest->id,
                'تم إنشاء طلب جديد لقطع الغيار.'
            );

            DB::commit();


            $customPartRequest->load(['model', 'category', 'items']);

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.custom_part_request_created'),
                'data' => new CustomPartRequestResource($customPartRequest),
                'settings' => [
                    'welcome_message' => 'hello',
                    'color_top_header' => 'part mart',
                ],
            ], 201);
        } catch (ValidationException $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'بيانات غير صالحة',
                'errors' => $e->errors(),
            ], 422);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating custom part request: ' . $e->getMessage(), [
                'request' => $request->all(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_create_custom_part_request'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function index(Request $request)
   
    {
        try {
            $userId = auth()->id();
            if (!$userId) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            $requests = CustomPartRequest::with(['model', 'category', 'sparePartOffers', 'items'])
                ->where('user_id', $userId)
                ->orderBy('created_at', 'desc')
                ->get();

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.custom_part_requests_retrieved'),
                'data' => CustomPartRequestResource::collection($requests),
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error retrieving custom part requests: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_retrieve_custom_part_requests'),
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function data()
   
    {
        $vendors = Vendor::where('type', 2)->get();
        return response()->json([
            'status' => 'success',
            'service_center' => VendorResource::collection($vendors),
            'brands' => BrandModelResource::collection(Brand::all()),
            'category' => CategoryResource::collection(Category::all())

        ], 200);
    }
    
    public function myRequests()
 
    {
            
             $request_with_no_offers = \App\Models\CustomPartRequest::where('user_id', auth()->id())
                            ->whereHas('items', function ($q) {
                                $q->whereDoesntHave('itemOffers');
                            })
                            ->with([
                                'model.brand',
                                'category',
                                'city',
                                'items' => function ($q) {
                                    $q->whereDoesntHave('itemOffers'); 
                                }
                            ])
                            ->get();
                
                
                
               $request_withoffer_pending = \App\Models\CustomPartRequest::where('user_id', auth()->id())
                            ->whereHas('items', function ($q) {
                                $q->whereHas('itemOffers',function($q){
                                    $q->where('status','pending');
                                });
                            }) 
                            ->with([
                                'model.brand',
                                'category',
                                'city',
                                'items' => function ($q) {
                                    $q->whereHas('itemOffers',function($q){
                                      $q->where('status','pending');
                                }); 
                                }
                            ])
                            ->get();
                  
                   $request_completed = \App\Models\CustomPartRequest::where('user_id', auth()->id())
                            ->where('status', 'pending')
                            ->whereHas('items.itemOffers', function ($q) {
                                $q->where('status', 'accepted');
                            })
                            ->with([
                                'model.brand',
                                'category',
                                'city',
                                'items' => function ($q) {
                                    $q->whereHas('itemOffers', function ($q) {
                                        $q->where('status', 'accepted');
                                    });
                                }
                            ])
                            ->get();
                  
                  
                  
                  $request_previouse = \App\Models\CustomPartRequest::where('user_id', auth()->id())
                            ->whereIn('status',['completed','prepared','shipped']) // Ensure at least one item has no offers
                            ->with([
                                'model.brand',
                                'category',
                                'city',
                                'items' => function ($q) {
                                    $q->whereHas('itemOffers',function($q){
                                      $q->where('status','accepted');
                                }); // Only items with no offers
                                }
                            ])
                            ->get();

     

        return $this->apiResponse(true, 'user requested retrieved successfully', [],[
           
            "request_current"=> CustomPartRequestResource::collection($request_with_no_offers),
            "request_wating"=> CustomPartRequestResource::collection($request_withoffer_pending),
            "request_previouse"=> CustomPartRequestResource::collection($request_previouse),
            "request_completed"=>CustomPartRequestResource::collection($request_completed),

        
        ]);

    }
  
    public function requestOfferAccept($id,Request $request)
    
    {
        $offer = SparePartOffer::findOrFail($id);
       
        try {
          
            \DB::beginTransaction();
          
            $offer->status = 'accepted';
            $offer->save();
            $vendorNotifier = new VendorNotificationService();

            $vendorNotifier->acceptOffer(
                $offer->vendor->id,
                $offer->id,
             );

            \DB::commit();

            return $this->apiResponse(true, __('messages.Offer accepted successfully'), [
               
            ], []);

        } catch (\Exception $e) {
            \DB::rollBack();
            return $this->apiResponse(false, 'فشل في قبول العرض', [], ['error' => $e->getMessage()]);
        }
    }
   
    public function check($code)
    {
        

        $coupon = Coupon::where('code', $code)->first();

        if (!$coupon) {
            return response()->json([
                'status' => false,
                'message' => 'هذا الكوبون غير موجود',
                'data' => [],
                'errors' => [],
            ], 404);
        }

        if (!$coupon->isValid()) {
            return response()->json([
                'status' => false,
                'message' => 'هذا الكوبون غير صالح أو منتهي',
                'data' => [],
                'errors' => [],
            ], 400);
        }

      return $coupon;
    }

  /*   public function completeRequestPay(Request $request){
            
           $validated = $request->validate([
                'request_id' => 'required',
                'copoun'=>'string',
           ]);
        
           $total_mount=null;
           $customRequest=CustomPartRequest::find($request->request_id);
           foreach($customRequest->items as $item){
             $total_mount=+$item->accepted_offer_price;
           }
           if($request->copoun){
              $coupon=Coupon::where('code',$request->copoun)->first();
              if($coupon->type == 'percentage'){
                  $total_mount=$total_mount-($total_mount*$coupon->value/100);

              }
              else{
                  $total_mount=$total_mount-$coupon->value;
              }
           }
           $customRequest->request_price=$total_mount;
           $customRequest->status='completed';
           $customRequest->save();
            return $this->apiResponse(true, __('messages.Request completed successfully'), [
               
            ], []);


    } */
    public function completeRequestPay(Request $request)
{
    $validated = $request->validate([
        'request_id' => 'required|exists:custom_part_requests,id',
        'copoun' => 'nullable|string',
    ]);

    $customRequest = CustomPartRequest::with('items.acceptedOffer')->findOrFail($request->request_id);

   
    $total_mount = 0;
    foreach ($customRequest->items as $item) {
        if ($item->acceptedOffer) {
            $total_mount += $item->acceptedOffer->price;
        }
    }

    if ($request->copoun) {
        $coupon = Coupon::where('code', $request->copoun)->first();
        if ($coupon) {
            if ($coupon->type == 'percentage') {
                $total_mount = $total_mount - ($total_mount * $coupon->value / 100);
            } else {
                $total_mount = $total_mount - $coupon->value;
            }
        }
    }

    
    $pendingItems = $customRequest->items()->whereDoesntHave('acceptedOffer')->get();

   
    if ($pendingItems->isNotEmpty()) {
        $newRequest = $customRequest->replicate();
        $newRequest->status = 'pending';
        $newRequest->request_price = null;
        $newRequest->save();

        foreach ($pendingItems as $item) {
            $item->custom_part_request_id = $newRequest->id;
            $item->save();
        }
    }

    // تحديث الطلب الحالي
    $customRequest->request_price = $total_mount;
    $customRequest->status = 'completed';
    $customRequest->save();

    return $this->apiResponse(true, __('messages.Request completed successfully'),[], [
        'request_id' => $customRequest->id,
        'new_pending_request_id' => $pendingItems->isNotEmpty() ? $newRequest->id : null,
        'total_paid' => $total_mount,
    ]);
}
   
   
    public function cancelledCustomeRequest($id)
    {
        $customRequest=CustomPartRequest::find($id);
        $customRequest->delete();
        return $this->apiResponse(true, __('messages.Request cancelled successfully'), [], []);

    }
}