<?php
namespace App\Http\Controllers\MobileApp;

use Exception;
use App\Models\Notification;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Services\FirebaseNotificationService;
use Illuminate\Support\Facades\Log;
use App\Traits\ApiTrait;
use App\Services\CustomFirebaseServive;
use Illuminate\Support\Facades\App;


class NotificationController extends Controller
{
   use ApiTrait;
    /**
     * Get user notifications
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */

   
    public function index(Request $request)
    {
        $user = Auth::user();
        $limit = $request->query('limit', 10); // Default to 10 notifications

        $notifications = $user->notifications()
            ->take($limit)
            ->get()
            ->map(function ($notification) {
                return [
                    'id' => $notification->id,
                    'type' => $notification->type,
                    'title'=>$notification->title,
                    'screen'=>$notification->screen,
                    'message' => $notification->message,
                    'read_at' => $notification->read_at,
                    'created_at' => $notification->created_at,
                ];
            });

        return response()->json([
            'status' => 'success',
            'message' => ['en' => 'Notifications retrieved successfully', 'ar' => 'تم استرجاع الإشعارات بنجاح'],
            'data' => $notifications,
        ], 200);
    }
    public function getUnreadCount(Request $request)
    {
        try {
            $user = Auth::user();

            if (!$user) {
                return response()->json([
                    'message' => 'Unauthenticated.',
                ], 401);
            }
            $unreadCount = Notification::where('user_id', $user->id)
                ->where('is_read', false)
                ->count();

            return response()->json([
                'message' => 'Unread notification count retrieved successfully.',
                'count' => $unreadCount,
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'message' => 'An error occurred while retrieving the notification count.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    public function markAllAsRead(Request $request)
    {
        try {
            $user = auth()->user();

            $user->notifications()
                    ->where('is_read', false)
                    ->update([
                        'is_read' => true,
                        'read_at' => now(),
                    ]);

            return response()->json([
                'status' => true,
                'message' => 'All notifications marked as read successfully.'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while marking notifications as read.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
    
    protected $firebaseService;

    public function __construct(FirebaseNotificationService $firebaseService,Request $request)
    {
         $locale = $request->header('Accept-Language', 'ar');
          App::setLocale($locale);
        $this->firebaseService = $firebaseService;
    }

  public function sendFirebaseTest(Request $request)
    {
        // dd('test');

        $request->validate([
            'user_id' => 'required|exists:users,id', 
            'title' => 'required|string',
            'body' => 'required|string', 
            'data' => 'nullable|array',
        ]);

        try {
            $user = User::findOrFail($request->user_id);
             
            $result = $this->firebaseService->sendNotification(
                $user,
                $request->title,
                $request->body,
                $request->data ?? []
            );

            if ($result) {
                return response()->json([
                    'status' => true,
                    'message' => 'Notification sent successfully to user.',
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Failed to send notification. Check logs for details.',
                ], 500);
            }
        } catch (\Exception $e) {
            Log::error('Firebase notification test failed.', ['error' => $e->getMessage()]);
            return response()->json([
                'status' => false,
                'message' => 'An error occurred while sending the notification.',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
   
    public function delete($id){
        $notification = Notification::find($id);
        $notification->delete();
        return $this->apiResponse(true, __('messages.Notification deleted successfully.'), [], []);
    }
   
    public function deleteAll(){
        $vendor = auth()->user();
        $vendor->notifications()->delete();
        return $this->apiResponse(true, __('messages.All notifications deleted successfully.'), [], []);
    }
    public function store(Request $request){
        $firebase = new CustomFirebaseServive();
        $firebase->send(null,$request->user_id,$request->title,$request->message,$request->type,$request->related_id,$request->related_type);
        return $this->apiResponse(true, 'All notifications add .', [], []);
    }

}