<?php

namespace App\Http\Controllers\MobileApp;

use App\Http\Controllers\Controller;
use App\Models\Order;
use App\Models\Cart;
use App\Models\OrderItem;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Http\Resources\GovernorateResource;
use App\Http\Resources\OrderSparpartResource;
class OrderController extends Controller
{

public function store(Request $request)
{
    $request->validate([
        'address' => 'required|string',
        'payment_method' => 'required|string|in:cash,credit_card,online',
        'total_price' => 'required|numeric|min:0',
        'phone_number'=>'required',
        'full_name'=>'required',
    ]);
    $user = auth()->user();
    // spicial order not from cart 
    if($request->items){
            // Create the order
             try {
                        $order = Order::create([
                            'user_id' => $user->id,
                            'status' => 'pending',
                            'total_price' => $request->total_price,
                            'shipping_address' => $request->address,
                            'payment_method' => $request->payment_method,
                            'phone' => $request->phone_number,
                            'full_name' => $request->full_name,
                        ]);

                        // Create order items and decrement stock
                        foreach ($request->items as $item) {
                            OrderItem::create([
                                'order_id' => $order->id,
                                'spare_part_id' => $item['product_id'], 
                                'quantity' => 1,
                                'unit_price' => $item['price'],
                                'has_discount' => false,
                                'discount_amount' => 0,
                            ]);
                           /// $item->sparePart()->decrement('stock', 1);
                        }
                        
                        return response()->json([
                            'status' => true,
                            'message' => 'تم إنشاء الطلب بنجاح.',
                            'data' => $order->load('items.sparePart')
                        ], 201);
                    }
                    catch (\Exception $e) {
                        \DB::rollBack();
                        return response()->json([
                            'status' => false,
                            'message' => 'حدث خطأ أثناء إنشاء الطلب.',
                            'error' => $e->getMessage(),
                        ], 500);
                    }
    }

  
   // an order from cart
    $cart = Cart::where('user_id', $user->id)->with('items.sparePart')->first();

    if (! $cart || $cart->items->isEmpty()) {
        return response()->json([
            'status' => false,
            'message' => 'السلة فارغة ولا يمكن إنشاء طلب.'
        ], 422);
    }

  
        try {
                // Check stock before creating order
                foreach ($cart->items as $item) {
                    $sparePart = $item->sparePart;
                    
                    if (is_null($sparePart->stock) || $sparePart->stock < $item->quantity) {
                        return response()->json([
                            'status' => false,
                            'message' => 'Quantity exceeds available stock for item ID: ' . $item->spare_part_id,
                            'item_id' => $item->spare_part_id,
                        ], 400);
                    }
                }

                // Create the order
                $order = Order::create([
                    'user_id' => $user->id,
                    'status' => 'pending',
                    'total_price' => $request->total_price,
                    'shipping_address' => $request->address,
                    'payment_method' => $request->payment_method,
                    'phone' => $request->phone_number,
                    'full_name' => $request->full_name,
                ]);

                // Create order items and decrement stock
                foreach ($cart->items as $item) {
                    OrderItem::create([
                        'order_id' => $order->id,
                        'spare_part_id' => $item->spare_part_id,
                        'quantity' => $item->quantity,
                        'unit_price' => $item->price,
                        'has_discount' => false,
                        'discount_amount' => 0,
                    ]);

                    $item->sparePart()->decrement('stock', $item->quantity);
                }

                // Clean up cart
                $cart->items()->delete();
                $cart->delete();

                \DB::commit();

                return response()->json([
                    'status' => true,
                    'message' => 'تم إنشاء الطلب بنجاح.',
                    'data' => $order->load('items.sparePart')
                ], 201);
    } catch (\Exception $e) {
        \DB::rollBack();
        return response()->json([
            'status' => false,
            'message' => 'حدث خطأ أثناء إنشاء الطلب.',
            'error' => $e->getMessage(),
        ], 500);
    }
}

public function index()
{
    $orders = Order::with('items.sparePart')
        ->where('user_id', auth()->id())
        ->latest()
        ->paginate(10); // change 10 to any per-page value

    return response()->json([
        'status' => true,
        'message' => 'تم استرجاع الطلبات بنجاح.',
        'data' => OrderSparpartResource::collection($orders),
        'meta' => [
            'current_page' => $orders->currentPage(),
            'last_page' => $orders->lastPage(),
            'per_page' => $orders->perPage(),
            'total' => $orders->total(),
        ]
    ]);
}
   
}