<?php
namespace App\Http\Controllers\MobileApp;

use App\Models\UserCar;
use App\Models\CarModel;
use App\Models\CarCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Http\Resources\ModelResource;
use Illuminate\Support\Facades\Cache;
use App\Http\Resources\UserCarResource;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use App\Http\Requests\UpdateUserCarRequest;
use App\Http\Resources\CarCategoryResource;
use App\Http\Requests\MobileApi\StoreUserCarRequest;
use App\Http\Resources\BrandResource;
use App\Models\Brand;

class UserCarController extends Controller
{
    /**
     * Get all user cars
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {
        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            $locale = \Illuminate\Support\Facades\App::getLocale();

           // $cars = Cache::remember('user_cars_' . $user->id . '_' . $locale, 3600, function () use ($user) {
                $cars=UserCar::with(['carModel', 'user'])
                    ->where('user_id', $user->id)
                    ->whereNull('deleted_at')
                    ->orderBy('id')
                    ->get();
          //  });

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.user_cars_retrieved'),
                'data' => UserCarResource::collection($cars),
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error retrieving user cars', [
                'user_id' => $user->id ?? null,
                'locale' => \Illuminate\Support\Facades\App::getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_retrieve_user_cars'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    /**
     * Store a new user car
     *
     * @param StoreUserCarRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        try {
            // Check if user is authenticated
            $user = Auth::guard('sanctum')->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            // Validate request data
            $validator = Validator::make($request->all(), [
                'model_id' => 'required', // Adjust table name if needed
                'manufacturing_year' => 'required|integer|min:1900|max:' . date('Y'),
                'engine_id' => 'nullable',
                'chassis_number' => 'nullable|string|max:17|unique:user_cars,chassis_number',
                'chassis_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048', // For file uploads
                'car_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048', // For file uploads
                //'name' => 'required|string|max:255',
                'description' => 'nullable|string|max:1000',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.validation_failed'),
                    'errors' => $validator->errors(),
                ], 422);
            }

            // Prepare data for creation
            $data = $validator->validated();
            $data['user_id'] = $user->id;

            // Handle file uploads
            if ($request->hasFile('chassis_image')) {
                $data['chassis_image'] = $request->file('chassis_image')->store('chassis_images', 'public');
            }
            if ($request->hasFile('car_image')) {
                $data['car_image'] = $request->file('car_image')->store('car_images', 'public');
            }

            DB::beginTransaction();
            // Create the UserCar record
            $car = UserCar::create($data);
            DB::commit();

            // Clear cache for user cars
            Cache::forget('user_cars_' . $user->id . '_en');
            Cache::forget('user_cars_' . $user->id . '_ar');
            return response()->json([
                'status' => 'success',
                'message' => trans('messages.user_car_created'),
                'data' => new UserCarResource($car),
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating user car', [
                'user_id' => isset($user) ? $user->id : null,
                'data' => $request->all(),
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_create_user_car'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    /**
     * Update a user car
     *
     * @param UpdateUserCarRequest $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update(Request $request, $id)
    {
        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            $car = UserCar::where('user_id', $user->id)
                ->whereNull('deleted_at')
                ->findOrFail($id);

            $data = $request->validated();

            DB::beginTransaction();

            $car->update($data);

            DB::commit();

            // Clear cache for user cars
            Cache::forget('user_cars_' . $user->id . '_en');
            Cache::forget('user_cars_' . $user->id . '_ar');

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.user_car_updated'),
                'data' => new UserCarResource($car),
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.user_car_not_found'),
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating user car', [
                'user_id' => $user->id ?? null,
                'car_id' => $id,
                'locale' => \Illuminate\Support\Facades\App::getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_update_user_car'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    /**
     * Delete a user car
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function destroy(Request $request, $id)
    {
        try {
            $user = auth('sanctum')->user();
            if (!$user) {
                return response()->json([
                    'status' => 'error',
                    'message' => trans('messages.unauthorized'),
                ], 401);
            }

            $car = UserCar::where('user_id', $user->id)
                ->whereNull('deleted_at')
                ->findOrFail($id);

            DB::beginTransaction();

            $car->delete();

            DB::commit();

            // Clear cache for user cars
            Cache::forget('user_cars_' . $user->id . '_en');
            Cache::forget('user_cars_' . $user->id . '_ar');

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.user_car_deleted'),
            ], 200);
        } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.user_car_not_found'),
            ], 404);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting user car', [
                'user_id' => $user->id ?? null,
                'car_id' => $id,
                'locale' => \Illuminate\Support\Facades\App::getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_delete_user_car'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    public function data(Request $request)
    {
        try {
            $locale = app()->getLocale();
            $cacheKey = 'models_' . md5(serialize($request->all()) . '_' . $locale);

            $models = Cache::remember($cacheKey, 3600, function () use ($request) {
                $query = CarModel::with(['brand', 'carCategory']);

                // Apply filters
                if ($request->has('brand_id')) {
                    $query->where('brand_id', $request->input('brand_id'));
                }
                if ($request->has('car_category_id')) {
                    $query->where('car_category_id', $request->input('car_category_id'));
                }

                return $query->orderBy('id')->get();
            });

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.models_retrieved'),
                'brands' => BrandResource::collection(Brand::all()),

                'categories_cars' => CarCategoryResource::collection(CarCategory::all()),
                'models' => ModelResource::collection($models),

            ], 200);
        } catch (\Exception $e) {
            Log::error('Error retrieving car models', [
                'query' => $request->all(),
                'locale' => app()->getLocale(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return response()->json([
                'status' => 'error',
                'message' => trans('messages.failed_to_retrieve_models'),
                'debug' => app()->environment('local') ? $e->getMessage() : null,
            ], 500);
        }
    }

    public function customeCar(Request $request){
        
        try{
              $user = auth('sanctum')->user();

             $validator = Validator::make($request->all(), [
                'brand' => 'required|string', 
                'model' => 'required|string',
                'manufacturing_year' => 'required|integer|min:1900|max:' . date('Y'),
                'engine_id' => 'nullable',
                'chassis_number' => 'nullable|string|max:17|unique:user_cars,chassis_number',
                'chassis_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048', // For file uploads
                'car_image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048', // For file uploads
                //'name' => 'required|string|max:255',
                'description' => 'nullable|string|max:1000',
                
            ]);
         
            if ($validator->fails()) {
                return response()->json([
                    'status' => false,
                    'errors' => $validator->errors()
                ], 422);
            }
            DB::beginTransaction();

             $brand=Brand::create([
               'name' => [
                    'ar' => $request->brand,
                    'en' => $request->brand,
                ],
             ]);

              $model=CarModel::create([
                'name' => [
                        'ar' => $request->model,
                        'en' => $request->model,
                    ],            
                        
                'brand_id'=>  $brand->id,
                'year'=> $request->manufacturing_year
             ]);
              // Prepare data for creation
            $data = $validator->validated();
            $data['user_id'] = $user->id;
            $data['model_id']=$model->id;

            // Handle file uploads
            if ($request->hasFile('chassis_image')) {
                $data['chassis_image'] = $request->file('chassis_image')->store('chassis_images', 'public');
            }
            if ($request->hasFile('car_image')) {
                $data['car_image'] = $request->file('car_image')->store('car_images', 'public');
            }
              
           
            // Create the UserCar record
            $car = UserCar::create($data);
            
            DB::commit();

            return response()->json([
                'status' => 'success',
                'message' => trans('messages.user_car_created'),
                'data' => new UserCarResource($car),
            ], 201);
          

        }
        catch (\Exception $e) {
        DB::rollBack();

        return response()->json([
            'status' => false,
            'message' => 'Something went wrong',
            'error' => $e->getMessage()
        ], 500);
    }
    }
}